from fastapi import FastAPI, Depends, HTTPException, BackgroundTasks
from sqlalchemy.orm import Session
import models, schemas, crud
from database import SessionLocal, engine
from email_utils import send_contact_email
from auth_utils import hash_password, verify_password
from fastapi.middleware.cors import CORSMiddleware
import shutil
import uuid
from fastapi import UploadFile, File, Form
from fastapi.staticfiles import StaticFiles

models.Base.metadata.create_all(bind=engine)
app = FastAPI()
app.mount("/uploads", StaticFiles(directory="uploads"), name="uploads")
app.add_event_handler("startup", lambda: print("Starting FastAPI application..."))
app.add_event_handler("shutdown", lambda: print("Shutting down FastAPI application..."))

app.add_middleware(
    CORSMiddleware,
    allow_origins=["*"],
    allow_methods=["*"],
    allow_headers=["*"],
)

def get_db():
    db = SessionLocal()
    try:
        yield db
    finally:
        db.close()

@app.post("/signup", response_model=schemas.UserOut)
def signup(user: schemas.UserCreate, db: Session = Depends(get_db)):
    existing = db.query(models.User).filter(models.User.email == user.email).first()
    if existing:
        raise HTTPException(status_code=400, detail="Email already registered")
    return crud.create_user(db, user)

@app.post("/login", response_model=schemas.UserOut)
def login(user: schemas.UserLogin, db: Session = Depends(get_db)):
    valid_user = crud.authenticate_user(db, user.email, user.password)
    if not valid_user:
        raise HTTPException(status_code=401, detail="Invalid email or password")
    return valid_user

@app.post("/contact", response_model=schemas.ContactOut)
async def contact_us(contact: schemas.ContactCreate, background_tasks: BackgroundTasks, db: Session = Depends(get_db)):
    saved = crud.submit_contact(db, contact)
    background_tasks.add_task(send_contact_email, contact.name, contact.email, contact.phone, contact.message)
    return saved




@app.post("/banner/upload", response_model=schemas.BannerOut)
def upload_banner(title: str = Form(...), image: UploadFile = File(...), db: Session = Depends(get_db)):
    file_ext = image.filename.split('.')[-1]
    file_name = f"{uuid.uuid4().hex}.{file_ext}"
    file_path = f"uploads/{file_name}"
    
    with open(file_path, "wb") as buffer:
        shutil.copyfileobj(image.file, buffer)
    
    banner = schemas.BannerCreate(title=title, image_url=file_path)
    return crud.create_banner(db, banner)


# Banner Routes
@app.post("/banner", response_model=schemas.BannerOut)
def add_banner(banner: schemas.BannerCreate, db: Session = Depends(get_db)):
    return crud.create_banner(db, banner)

@app.get("/banner", response_model=list[schemas.BannerOut])
def list_banners(db: Session = Depends(get_db)):
    return crud.get_banners(db)

@app.delete("/banner/{id}")
def delete_banner(id: int, db: Session = Depends(get_db)):
    crud.delete_banner(db, id)
    return {"message": "Banner deleted"}

@app.post("/gallery/upload", response_model=schemas.GalleryOut)
def upload_gallery(description: str = Form(...), image: UploadFile = File(...), db: Session = Depends(get_db)):
    file_ext = image.filename.split('.')[-1]
    file_name = f"{uuid.uuid4().hex}.{file_ext}"
    file_path = f"uploads/{file_name}"
    
    with open(file_path, "wb") as buffer:
        shutil.copyfileobj(image.file, buffer)
    
    gallery = schemas.GalleryCreate(description=description, image_url=file_path)
    return crud.create_gallery(db, gallery)

# Gallery Routes
@app.post("/gallery", response_model=schemas.GalleryOut)
def add_gallery(gallery: schemas.GalleryCreate, db: Session = Depends(get_db)):
    return crud.create_gallery(db, gallery)

@app.get("/gallery", response_model=list[schemas.GalleryOut])
def list_gallery(db: Session = Depends(get_db)):
    return crud.get_gallery(db)

@app.delete("/gallery/{id}")
def delete_gallery(id: int, db: Session = Depends(get_db)):
    crud.delete_gallery(db, id)
    return {"message": "Gallery item deleted"}

@app.post("/service/upload", response_model=schemas.ServiceOut)
def upload_service(
    title: str = Form(...),
    description: str = Form(...),
    image: UploadFile = File(...),
    db: Session = Depends(get_db)
):
    file_ext = image.filename.split('.')[-1]
    file_name = f"{uuid.uuid4().hex}.{file_ext}"
    file_path = f"uploads/{file_name}"
    with open(file_path, "wb") as buffer:
        shutil.copyfileobj(image.file, buffer)

    image_url = f"/uploads/{file_name}"
    service = schemas.ServiceCreate(title=title, description=description, image_url=image_url)
    return crud.create_service(db, service)


@app.get("/services", response_model=list[schemas.ServiceOut])
def list_services(db: Session = Depends(get_db)):
    return crud.get_services(db)

@app.put("/service/{service_id}", response_model=schemas.ServiceOut)
def update_service(service_id: int, update: schemas.ServiceUpdate, db: Session = Depends(get_db)):
    service = crud.update_service(db, service_id, update)
    if not service:
        raise HTTPException(status_code=404, detail="Service not found")
    return service

@app.delete("/service/{service_id}")
def delete_service(service_id: int, db: Session = Depends(get_db)):
    service = crud.delete_service(db, service_id)
    if not service:
        raise HTTPException(status_code=404, detail="Service not found")
    return {"message": "Service deleted"}
